/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  Image.h - FW image manipulation class
 *
 *  Version: $Id: TImage.h 2752 2006-01-19 14:40:17Z mst $
 *
 */

#ifndef TIMAGE_H
#define TIMAGE_H

#include "Image.h"

// Boot section names:

#define BOOT0_EXE_NAME  "failsafe.exe"
#define BOOT2_EXE_NAME  "boot2.exe"
#define BOOT3_EXE_NAME  "boot2.exe_boot3"

// Parameter which are referenced by name in MIC: names used:

#define VSD_PARAM_NAME  "ADAPTER.adapter_vsd"
#define PSID_PARAM_REFNAME "PSID"

#define BOOT0_LOAD_NAME "Auto.boot0_load"

#define PLL_PARAM_MAMES    {"PLL.PLL0", "PLL.PLL1", "PLL.PLL2", "PLL.PLL3" }
#define GUID_PARAM_NAMES_OLD  { "IB.Node GUID [63:32]",       "IB.Node GUID [31:0]",    \
			     "IB.Port1 GUID [63:32]",         "IB.Port1 GUID [31:0]",   \
                             "IB.Port2 GUID [63:32]",         "IB.Port2 GUID [31:0]",   \
                             "IB.SystemImageGUID [63:32]",    "IB.SystemImageGUID [31:0]" }

#define GUID_PARAM_NAMES  {  "IB.nodeguid_63_32",             "IB.nodeguid_31_0",       \
			     "IB.port1guid_63_32",            "IB.port1guid_31_0",      \
                             "IB.port2guid_63_32",            "IB.port2guid_31_0",      \
                             "IB.systemimageguid_63_32",      "IB.systemimageguid_31_0" }
			     


// FW._IS_RevisionID
// FW._log2_sector_sz_ptr
// FW._RevisionID --> optional. In not exist Assumed to be string "a0" .

// FW._Is_Flash
// FW._Flash_SectorSize
// FW._Flash_nBanks
// FW._Flash_SizeIdx

class TImage : public Image {
public:
    TImage(ParamList   *plist, 
           const char* exp_rom_file, 
           const char* user_data_file, 
           const char* fw_conf_file,
           bool        add_info_sect): 
	Image(plist,true), 
	_im_total(0), 
	_total(0),  
	_current_written_addr(0),
	_exp_rom_fn(exp_rom_file),
	_user_data_fn(user_data_file),
	_fw_conf_fn(fw_conf_file),
        _add_info_sect(add_info_sect)    {};

    ~TImage();

    enum {
	NPLLS  = 4, 
	NGUIDS = 8
    };

    enum SectionType { 
	H_FIRST      = 1,    
	H_DDR        = 1, 
	H_CNF        = 2, 
	H_JMP        = 3, 
	H_EMT        = 4, 
	H_ROM        = 5, 
	H_GUID       = 6,
	H_BOARD_ID   = 7,
	H_USER_DATA  = 8,
	H_FW_CONF    = 9,
        H_IMG_INFO   = 10,
	H_LAST       = 10
    };

    virtual bool param_check      (ParamList& plist);
    virtual bool compile          (ParamList &plist,  const MultiFile &mfile);
//--    virtual bool write            (const char *fname, const char *format_name=0);
    
    virtual bool get_fw_ver       (FwVer v);

    bool         CreateImage      (const MultiFile& mfile);
    
protected:

    typedef      std::list<ImSection*>  SectList;    
    
    virtual
       bool      write_image      (FILE* fp, const char* fname, FORMATS frm);

    bool         mlx_dump         (FILE* fp, u_int32_t rom_offset, SectList& sects); // !!! Move this func to Image and unify with Anafa
    bool         binary_dump      (FILE* fp, u_int32_t rom_offset, SectList& sects);

    bool         ArrangeAddresses ();
    bool         add_sects        (SectList& sl, u_int32_t sect_size = 0);


    bool         SetReplaces      ();
    bool         fs_info_dump     (FILE* fp);
    bool         set_crc_rep      (ImSection *s, u_int32_t offs);

    //////////////////////////////////
    //
    //   Sections Types classes
    //
    //////////////////////////////////


    class VbhSect : public ImSection {
	public:
	VbhSect(const std::string& name);

	virtual bool    Pack         (std::vector<u_int8_t>& data);
	virtual int     Size         ()     { return HdrSize();}  
	virtual int     HdrSize      ()     { return 40;}
	
	virtual bool    SetNext      (u_int32_t nptr)        {nptr = 0; return true;} 

//--	u_int32_t       GetStart() { return strt_addr;} 

	u_int32_t       pll[NPLLS]; // Bytes: 0x00 - 0x0f
        u_int32_t       id[4];      // Bytes: 0x10 - 0x1f
        u_int32_t       tlen;       // Bytes: 0x20 - 0x27   ????
        u_int32_t       guid_ptr;   // Bytes: 0x2a - 0x2d   ????

	bool            CreateVbhSect(ParamList* plist);
	
    protected:
	void            clear();

	bool            fill_pll     (ParamList* plist);
	bool            fill_id      (ParamList* plist);
	bool            get_fwid     (ParamList* plist, u_int32_t &fid);

	static 
	  const char*   _pll_names[NPLLS];
    };

    class BootSect : public ImSection {
	public:
	BootSect(const std::string& name) : ImSection(name) {} 
   
	virtual bool    Pack         (std::vector<u_int8_t>& data);
	virtual int     HdrSize      ()                      {return 12;} 
	virtual bool    SetNext      (u_int32_t nptr)        {nptr = 0; return true;} 

	bool            CreateBootSect(MSection* s);

	u_int32_t       code_addr;
	u_int32_t       code_size;
	u_int32_t       strt_addr;

    };

    class PtrSect  : public ImSection {
	public:
	PtrSect(const std::string& name);

	virtual bool    Pack         (std::vector<u_int8_t>& data);
	virtual int     HdrSize      ()               {return sizeof(u_int32_t)*4 + sizeof(fw_reserved) + sizeof(vsd) +sizeof(psid);}

	bool            CreatePtrSect(u_int32_t addr, u_int32_t size, ParamList* plist);

	enum { 
	    SIGNATURE        = 0x5a445a44, 
	    SIGNATURE_OFFSET = 8 ,
	    PSID_OFFSET      = 240,
	    VSD_OFFSET       = 32  
	};

	u_int32_t       fi_addr;
	u_int32_t       fi_size;
	u_int32_t       signature;
	u_int32_t       fw_reserved[5];
	char            vsd[208];
	char            psid[16];
	u_int32_t       branch_to;
    };


    class GpSect : public ImSection {
	public:
	GpSect(const std::string& name) : ImSection(name) {}

        // 1	8	FW Version
        // 2	8	FW Build Time
        // 3	4	Device Type
        // 4	16	PSID
        // 5	256	VSD
        // 6	Varying	Supported PSIDs (TBD)
        // 0xFF	0	END
        // All others		Reserved


        enum ImageInfoTags {
            II_IiFormatRevision   = 0,
            II_FwVersion          = 1,
            II_FwBuildTime        = 2, 
            II_DeviceType         = 3,
            II_PSID               = 4,
            II_VSD                = 5,
            II_SuppurtedPsids     = 6,
            II_End                = 0xff
        };

	virtual bool    Pack         (std::vector<u_int8_t>& data);
	virtual int     HdrSize      ()                        {return 16;} 
	virtual bool    SetNext      (u_int32_t nptr);

	bool            CreateExe    (MSection* s);
	bool            CreateCnf    (std::vector<u_int32_t>& cr_inits);
	bool            CreateGuids  (ParamList* plist, int gnum, u_int32_t guid_addr );
	bool            CreateJump   (const MultiFile& mfile);
	bool            CreateFile   (const std::string& file_name, SectionType file_type); 

        bool            CreateImageInfo(Image* img, ParamList* plist);

    private:
	bool            LoadImg(const std::string& file_name);
	bool            LoadBin(const std::string& file_name);

    public:
        u_int32_t       type;
	u_int32_t       size;
        u_int32_t       param;
        u_int32_t       next;
    };


    SectList   _invariant_sects;
    SectList   _image_sects;

    u_int32_t  _im_total;
    u_int32_t  _im_total_s;
    u_int32_t  _total;
    bool       _boot0_found;
    
    u_int32_t  _current_written_addr;

    const char* _exp_rom_fn;
    const char* _user_data_fn;
    const char* _fw_conf_fn;

    bool        _add_info_sect;

    GpSect*    _guid_sect;
};


#endif

